// $Id: CControl.cpp,v 1.6 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CControl.hpp"
#include <Exceptions/CException.hpp>
#include <IO/CTextStream.hpp>
#include "../Graphics/CImageManager.hpp"
using Exponent::GUI::Graphics::CImageManager;
using Exponent::IO::CTextStream;
using Exponent::Exceptions::CException;
using Exponent::GUI::Controls::CControl;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION_NO_PARENT(CControl);

//	===========================================================================
CControl::CControl(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener)
		: m_referenceCount(0)
		, m_doDefaultDrawing(false)
		, m_actionListener(listener)
		, m_rootControl(NULL)
		, m_parentControl(NULL)
		, m_mouseEnabled(true)
		, m_keyboardEnabled(false)
		, m_dropEnabled(false)
		, m_uniqueId(uniqueId)
		, m_value(1.0)
		, m_enabled(true)
		, m_needsRightClick(false)
		, m_primaryImage(NULL)
		, m_disabledImage(NULL)
		, m_frameColour(CAlphaColour::CALPHACOLOUR_BLACK)
		, m_backgroundColour(CAlphaColour::CALPHACOLOUR_WHITE)
{
	EXPONENT_CLASS_CONSTRUCTION(CControl);

	// Store the control root
	this->setControlRoot(root);

	// Store the area
	this->setArea(area);
}

//	===========================================================================
CControl::~CControl()
{
	EXPONENT_CLASS_DESTRUCTION(CControl);

	// NULL our pointers
	NULL_POINTER(m_actionListener);
	NULL_POINTER(m_rootControl);
	NULL_POINTER(m_parentControl);

	// Forget the primary and disabled images
	FORGET_COUNTED_OBJECT(m_primaryImage);
	FORGET_COUNTED_OBJECT(m_disabledImage);
}

//	===========================================================================
void CControl::setControlRoot(IControlRoot *controlRoot)
{
	// Check its valid
	if (controlRoot == NULL)
	{
		throw CException("Control root is null", "CControl::setControlRoot(IControlRoot *)");
	}

	// Store it
	m_rootControl = controlRoot;
}

//	===========================================================================
void CControl::setArea(const CRect &area)
{
	// Store the area
	m_area = area;
	m_normalisedArea.setRect(CPoint(0, 0), area.getDimension());

	// Store the absolute position
	CPoint position;
	m_rootControl->getWindowCoordinatesOfControl(this, position);
	m_absoluteArea.setRect(position, area.getDimension());

	// Store the dimension
	m_dimension = area.getDimension();
}

//	===========================================================================
void CControl::update()
{
	m_rootControl->updateControl(this);
}

//	===========================================================================
void CControl::drawControl(CGraphics &graphics)
{
	// Sub classes should handle this
}

//	===========================================================================
bool CControl::drawEnabledControl(CGraphics &graphics)
{
	// First we check if we are disabled and there is an image to draw
	if (!m_enabled && m_disabledImage)
	{
		// Draw the image
		graphics.drawImage(m_disabledImage, m_normalisedArea, m_disabledImage->getNormalisedImageSize());
	}

	// We want to let them know if they should draw
	return m_enabled;
}

//	===========================================================================
bool CControl::drawPrimaryImage(CGraphics &graphics)
{
	// Check there is a primary image
	if (m_primaryImage)
	{
		// Draw the image
		graphics.drawImage(m_primaryImage, m_normalisedArea, m_primaryImage->getNormalisedImageSize());

		// Sucess!
		return true;
	}

	// Failed as no image
	return false;
}

//	===========================================================================
void CControl::drawPrimaryImage(CGraphics &graphics, const bool drawFrameIfNoImage)
{
	// Check there is a primary image
	if (m_primaryImage)
	{
		// Draw the image
		graphics.drawImage(m_primaryImage, m_normalisedArea, m_primaryImage->getNormalisedImageSize());
	}
	else if (drawFrameIfNoImage)
	{
		// Else they want us to draw the default square that we hold

		// Draw the fill area
		graphics.getMutableBrush()->setColour(m_backgroundColour);
		graphics.fillRectangle(m_normalisedArea);

		// Draw the surround
		graphics.getMutablePen()->setColour(m_frameColour);
		graphics.drawRectangle(m_normalisedArea);
	}
}

//	===========================================================================
void CControl::enableControl()
{
	m_enabled = true;
}

//	===========================================================================
void CControl::disableControl()
{
	m_enabled = false;
}

//	===========================================================================
void CControl::setPrimaryImage(IImage *theImage)
{
	EXCHANGE_COUNTED_OBJECTS(m_primaryImage, theImage);
}

//	===========================================================================
void CControl::setDisabledImage(IImage *theImage)
{
	EXCHANGE_COUNTED_OBJECTS(m_disabledImage, theImage);
}

//	===========================================================================
void CControl::setDefaultControlColours(const CAlphaColour &backgroundColour, const CAlphaColour &frameColour)
{
	m_frameColour      = frameColour;
	m_backgroundColour = backgroundColour;
}

//	===========================================================================
bool CControl::serializeToNode(CXMLNode &node)
{
	CString rect;
	CString frameColour;
	CString backgroundColour;
	rect.setStringWithFormat("%li, %li, %li, %li", m_area.getLeft(), m_area.getTop(), m_area.getWidth(), m_area.getHeight());
	frameColour.setStringWithFormat("%li, %li, %li, %li", m_frameColour.getRed(), m_frameColour.getGreen(), m_frameColour.getBlue(), m_frameColour.getAlpha());
	backgroundColour.setStringWithFormat("%li, %li, %li, %li", m_backgroundColour.getRed(), m_backgroundColour.getGreen(), m_backgroundColour.getBlue(), m_backgroundColour.getAlpha());

	//node.setNo
	node.addAttribute(new CXMLAttribute("control_is_mouse_enabled",    m_mouseEnabled));
	node.addAttribute(new CXMLAttribute("control_is_key_enabled",      m_keyboardEnabled));
	node.addAttribute(new CXMLAttribute("control_is_drop_enabled",     m_dropEnabled));
	node.addAttribute(new CXMLAttribute("control_needs_right_click",   m_needsRightClick));
	node.addAttribute(new CXMLAttribute("control_is_enabled",          m_enabled));
	node.addAttribute(new CXMLAttribute("control_should_default_draw", m_doDefaultDrawing));
	node.addAttribute(new CXMLAttribute("control_unique_id",		   m_uniqueId));
	node.addAttribute(new CXMLAttribute("control_value",			   m_value));
	node.addAttribute(new CXMLAttribute("control_area",				   rect));
	node.addAttribute(new CXMLAttribute("control_primary_image",	   CImageManager::getReferenceNameForImage(m_primaryImage)));
	node.addAttribute(new CXMLAttribute("control_disabled_image",	   CImageManager::getReferenceNameForImage(m_disabledImage)));
	node.addAttribute(new CXMLAttribute("control_frame_colour",		   frameColour));
	node.addAttribute(new CXMLAttribute("control_background_colour",   backgroundColour));

	return false;
}

//	===========================================================================
bool CControl::serializeFromNode(CXMLNode &node)
{
	return false;
}

//	===========================================================================
void CControl::referenced()
{ 
	m_referenceCount++; 
}

//	===========================================================================
void CControl::dereference()
{
	m_referenceCount--;
	if (m_referenceCount <= 0)
	{
		delete this;
	}
}

//	===========================================================================
void CControl::getObjectDescription(char *string, const long size) const
{
	strncpy(string, "Control", size);
}